/**
 * \file: daemon.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: Key Handling Daemon
 *
 * \author: Ian Molton (ian.molton@codethink.co.uk)
 *	   Norbert Uetrecht (nuetrecht@de.adit-jv.com)
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#define _GNU_SOURCE     /**< needed for 'struct ucred' */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/epoll.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdbool.h>
#include <stdint.h>
#include <libgen.h>
#include <linux/limits.h>
#include <systemd/sd-daemon.h>
#include <dirent.h>
#include <keyman.h>
#include <sdc.h>
#include <private/sdc_advanced.h>
#include <sdc/daemon/arch/sdc_daemon.h>

#include "daemon.h"
#include "daemon_log.h"
#include "keystore.h"
#include "key_container.h"
#include "helper.h"

#define MAX_EVENTS  64
#define UNIX_PATH_MAX    108

#define UMASK_SOCKET (S_IXOTH | S_IXGRP | S_IXUSR)

#undef NO_FORK          /**< Define to prevent forking as daemon */

/* catch_signal - signal handler */
static volatile int terminate = 0;
static volatile int terminate_with_err = 0;

static void catch_signal(int signo)
{
    switch (signo) {
    case SIGINT:
        terminate_with_err = 1;
    /* fall-through */
    case SIGQUIT:
    case SIGTERM:
        daemon_log(DAEMON_LOG_ERROR, "Catched signal, shutting down");
        terminate = 1;
        break;
    case SIGPIPE:
        break;
    case SIGUSR1:
        break;
    case SIGUSR2:
        break;
    }
}

/* install_signal_handler - install a handler for some signals */
static void install_signal_handler()
{
    int signals[] = { SIGINT, SIGQUIT, SIGTERM, SIGPIPE, SIGUSR1, SIGUSR2, 0 };
    unsigned i;
    struct sigaction sa;

    /* Install a signal handler for the above listed signals */
    for (i = 0; signals[i]; i++) {
        memset(&sa, 0, sizeof(sa));
        sa.sa_handler = catch_signal;
        if (sigaction(signals[i], &sa, NULL) < 0)
            daemon_log(DAEMON_LOG_CRITICAL, "%s %u",
                       "failed to install signal handler ", signals[i]);
    }
}

/**
 * Initialize the key store and listening socket paths
 *
 * \param key_store	Updated to path to use for key_store
 * \param listen	Updated to path to use for server socket
 * \return true on success, else false.
 */
static sdc_error_t init_paths(char **key_store, char **listen_sock)
{
    sdc_error_t err;
    char *path;

    *key_store = NULL;
    *listen_sock = NULL;

    /* Lookup in config file */
    err = sdc_config_file_lookup_absolute_path(&path, SDC_CONFIG_KEYSTORE);
    if (err != SDC_OK) {
        daemon_log(DAEMON_LOG_ERROR, "Invalid keystore config\n");
        return err;
    }

    *key_store = strndup(path, strlen(path));
    daemon_log(DAEMON_LOG_INFO, "Key store path: %s.\n", *key_store);

    free(path);

    /* Lookup in config file */
    err = sdc_config_file_lookup_absolute_path(&path, SDC_CONFIG_DAEMON_SOCKET);
    if (err != SDC_OK) {
        daemon_log(DAEMON_LOG_ERROR, "Invalid socket config\n");
        free(*key_store);
        return err;
    }

    *listen_sock = strndup(path, strlen(path));
    daemon_log(DAEMON_LOG_INFO, "Server listen path: %s.\n", *listen_sock);

    free(path);

    return SDC_OK;
}


/**
 * Log a request
 *
 * Both msg_install and key_data will be NULL when the operation type given in
 * the msg_header indicates MSG_REMOVE_KEY operation.  For the MSG_INSTALL_KEY operation,
 * msg_remove will be NULL.
 *
 * \param msg_header	    Header of request message
 * \param msg_uid           UID of the key to install/import
 * \param msg_install	    Install-specific part of message, or NULL
 * \param msg_import        Import-specific part of message, or NULL
 * \param msg_remove	    Remove-specific part of message, or NULL
 */
void log_request(const sdc_daemon_msg_header_type *msg_header,
                 uid_t msg_uid,
                 const sdc_daemon_msg_install_type *msg_install,
                 const sdc_daemon_msg_import_type *msg_import,
                 const sdc_daemon_msg_remove_type *msg_remove)
{
    gid_t gid;
    sdc_error_t err;
    bool volatile_key;
    bool random_key;
    size_t keylen;

    switch (msg_header->operation) {
    case MSG_INSTALL_KEY:
        if (msg_install == NULL) {
            daemon_log(DAEMON_LOG_ERROR, "No INSTALL msg data.");
            return;
        }

        volatile_key = msg_install->flags & MSG_FLAG_VOLATILE;
        random_key = msg_install->flags & MSG_FLAG_RANDOM_KEY;

        err = sdc_permissions_get_gid(&msg_install->permissions, &gid);
        if (err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR, "Failed to read gid from INSTALL msg data.");
            return;
        }

        sdc_key_len_get_bits(msg_install->len, &keylen);

        daemon_log(DAEMON_LOG_DEBUG,
                   "Received INSTALL msg for %s %s storage key (uid %u, gid %u - len %d)",
                   volatile_key ? "volatile" : "persistent",
                   random_key ? "random" : "plain",
                   msg_uid, gid,
                   keylen);

        if (msg_install->req_id_min == msg_install->req_id_max) {
            daemon_log(DAEMON_LOG_DEBUG,
                       "ID \"%u\" requested for key",
                       msg_install->req_id_min);
        } else {
            daemon_log(DAEMON_LOG_DEBUG,
                       "ID in range of \"%u\" to \"%u\" requested for key",
                       msg_install->req_id_min,
                       msg_install->req_id_max);
        }
        break;

    case MSG_IMPORT_KEY:
        if (msg_import == NULL) {
            daemon_log(DAEMON_LOG_ERROR, "No IMPORT msg data.");
            return;
        }

        volatile_key = msg_import->flags & MSG_FLAG_VOLATILE;

        err = sdc_permissions_get_gid(&msg_import->permissions, &gid);
        if (err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR, "Failed to read gid from INSTALL msg data.");
            return;
        }

        sdc_key_len_get_bits(msg_import->len, &keylen);

        daemon_log(DAEMON_LOG_DEBUG,
                   "Received IMPORT msg for %s storage key (uid %u, gid %u - len %d)",
                   volatile_key ? "volatile" : "persistent",
                   msg_uid, gid,
                   keylen);

        if (msg_import->req_id_min == msg_import->req_id_max) {
            daemon_log(DAEMON_LOG_DEBUG,
                       "ID \"%u\" requested for key",
                       msg_import->req_id_min);
        } else {
            daemon_log(DAEMON_LOG_DEBUG,
                       "ID in range of \"%u\" to \"%u\" requested for key",
                       msg_import->req_id_min,
                       msg_import->req_id_max);
        }
        break;

    case MSG_REMOVE_KEY:
        if (msg_remove == NULL) {
            daemon_log(DAEMON_LOG_ERROR, "No REMOVE msg data.");
            return;
        }
        daemon_log(DAEMON_LOG_DEBUG,
                   "Received REMOVE msg for key ID: %u.",
                   msg_remove->id);
        break;
    }
}


/**
 * Perform install/remove key operation
 *
 * Both msg_install and key_data will be NULL when the operation type given in
 * the msg_header indicates REMOVE operation.  For the INSTALL operation,
 * msg_remove will be NULL.
 *
 * \param msg_header	Header of request message
 * \param msg_install	Install-specific part of message, or NULL
 * \param msg_import    Import-specific part of message, or NULL
 * \param msg_remove	Remove-specific part of message, or NULL
 * \param key_data	Key data to install, or NULL
 * \param ctx		Libkeyman context
 * \param id		Updated to ID of key installed or removed.
 * \param cr		Credentials of process at client end of socket.
 * \return appropriate error value.  If KEYMAN_ERROR, consult key_err.
 */
sdc_error_t perform_op(const sdc_daemon_msg_header_type *msg_header,
                       const sdc_daemon_msg_install_type *msg_install,
                       const sdc_daemon_msg_import_type *msg_import,
                       const sdc_daemon_msg_remove_type *msg_remove,
                       uint8_t *key_data, keystore_ctx_t *ctx,
                       sdc_key_id_t *id, const struct ucred *cr)
{
    daemon_credentials_t cred;
    sdc_error_t sdc_err = SDC_OK;
    libkeyman_key_spec_t key_spec;
    bool volatile_key;

    cred.uid = cr->uid;
    cred.gid = cr->gid;

    if (msg_header == NULL) {
        daemon_log(DAEMON_LOG_ERROR, "No message header");
        return SDC_DAEMON_REQUEST_REJECTED;
    }

    switch (msg_header->operation) {
    case MSG_INSTALL_KEY:
        if (msg_install == NULL) {
            daemon_log(DAEMON_LOG_ERROR, "No install message");
            return SDC_DAEMON_REQUEST_REJECTED;
        }

        volatile_key = msg_install->flags & MSG_FLAG_VOLATILE;

        if (msg_install->flags & MSG_FLAG_RANDOM_KEY) {
            if (msg_install->key_data_len != 0)
                sdc_err = SDC_INVALID_PARAMETER;
        } else {
            if (msg_install->key_data_len == 0)
                sdc_err = SDC_INVALID_PARAMETER;
        }

        key_spec.fmt = msg_install->fmt;
        key_spec.len = msg_install->len;
        key_spec.permissions = &msg_install->permissions;

        /*
         * fmt and len of key are checked inside libkeyman.
         * Permission ranges are checked by libkeyman too.
         * Anyway permission uid needs to be checked here
         */
        if ((sdc_err == SDC_OK) && (msg_install->permissions.uid != cred.uid)) {
            sdc_err = SDC_PERM_INVALID;
        }

        if (sdc_err == SDC_OK) {
            sdc_err = keystore_install_plain_or_random_key(ctx,
                                                           msg_install->req_id_min, msg_install->req_id_max,
                                                           id,
                                                           &key_spec,
                                                           !volatile_key,
                                                           msg_install->key_data_enc, key_data, msg_install->key_data_len);
        }

        if (sdc_err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR, "keyman_install_key() "
                       "error: %s", sdc_get_error_string(sdc_err));
            return sdc_err;
        }
        break;

    case MSG_IMPORT_KEY:
        if (msg_import == NULL) {
            daemon_log(DAEMON_LOG_ERROR, "No import message");
            return SDC_DAEMON_REQUEST_REJECTED;
        }

        volatile_key = msg_import->flags & MSG_FLAG_VOLATILE;

        if (msg_import->key_data_len == 0)
            sdc_err = SDC_INVALID_PARAMETER;

        key_spec.fmt = msg_import->fmt;
        key_spec.len = msg_import->len;
        key_spec.permissions = &msg_import->permissions;

        /*
         * fmt and len of key are checked inside libkeyman.
         * Permission ranges are checked by libkeyman too.
         * Anyway permission uid needs to be checked here
         */
        if ((sdc_err == SDC_OK) && (msg_import->permissions.uid != cred.uid)) {
            sdc_err = SDC_PERM_INVALID;
        }

        if (sdc_err == SDC_OK) {
            sdc_err = keystore_import_key(ctx,
                                          msg_import->req_id_min, msg_import->req_id_max,
                                          id,
                                          &key_spec,
                                          !volatile_key,
                                          msg_import->key_data_enc, key_data, msg_import->key_data_len);
        }

        if (sdc_err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR, "keyman_import_key() "
                       "error: %s", sdc_get_error_string(sdc_err));
            return sdc_err;
        }
        break;

    case MSG_REMOVE_KEY:
        if (msg_remove == NULL) {
            daemon_log(DAEMON_LOG_ERROR, "No remove message");
            return SDC_DAEMON_REQUEST_REJECTED;
        }

        *id = msg_remove->id;

        sdc_err = keystore_remove_key(ctx, msg_remove->id, &cred);
        if (sdc_err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR, "keyman_remove_key() "
                       "error: %s", sdc_get_error_string(sdc_err));
            return sdc_err;
        }
        break;
    default:
        return SDC_INTERNAL_ERROR;
    }

    return SDC_OK;
}


/**
 * Handle client requests
 *
 * Both msg_install and key_data will be NULL when the operation type given in
 * the msg_header indicates REMOVE operation.  For the INSTALL operation,
 * msg_remove will be NULL.
 *
 * \param cs_fd		File descriptor for client socket
 * \param ctx		Context for libkeyman
 */
void sdc_handle_socket_request(int cs_fd, keystore_ctx_t *ctx)
{
    sdc_daemon_msg_header_type *msg_header = NULL;
    sdc_daemon_msg_install_type *msg_install = NULL;
    sdc_daemon_msg_import_type *msg_import = NULL;
    sdc_daemon_msg_remove_type *msg_remove = NULL;
    sdc_daemon_msg_response_type msg_response;
    uint8_t *key_data = NULL;
    sdc_error_t sdc_err = SDC_OK;
    socklen_t cr_len;
    struct ucred cr;
    sdc_key_id_t id = 0;
    int err_chk;
    uint8_t *msg_buf;
    size_t msg_buf_len;
    ssize_t received_len;
    ssize_t send_len;
    size_t expected_len;

    /* Get credentials of process on other end of socket */
    cr_len = sizeof(struct ucred);
    err_chk = getsockopt (cs_fd, SOL_SOCKET, SO_PEERCRED, &cr, &cr_len);
    if (err_chk == -1) {
        daemon_log(DAEMON_LOG_ERROR,
                   "getsockopt() errno: %u.", errno);
        return;
    }

    daemon_log(DAEMON_LOG_INFO, "Connection from process with "
               "UID: %u, GID: %u.", cr.uid, cr.gid);

    /* Max size of message */
    msg_buf_len = SDC_DAEMON_MAX_MSG_LEN;

    /* Allocate a buffer for the data */
    msg_buf = malloc(msg_buf_len);
    if (msg_buf == NULL) {
        sdc_err = SDC_NO_MEM;
        goto out;
    }

    /* Read message into buffer, and get its actual length */
    received_len = recv(cs_fd, msg_buf, msg_buf_len, 0);
    if (received_len == -1) {
        if (errno != EAGAIN && errno != EWOULDBLOCK) {
            daemon_log(DAEMON_LOG_ERROR, "read() errno: %u.", errno);
            sdc_err = SDC_DAEMON_COMMUNICATION_ERROR;
            goto out;
        }
    }

    /* Validate input */
    if (received_len <= 0) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Received empty message on socket.");
        sdc_err = SDC_DAEMON_REQUEST_REJECTED;
        goto out;

    } else if ((size_t)received_len < sizeof(sdc_daemon_msg_header_type)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Received too-short message on socket.");
        sdc_err = SDC_DAEMON_REQUEST_REJECTED;
        goto out;
    }

    /* Parse the message from the client */
    msg_header = (sdc_daemon_msg_header_type *) msg_buf;

    /* Validate the header */
    if (msg_header->msg_len != (size_t)received_len) {
        daemon_log(DAEMON_LOG_ERROR, "Message length mismatch: "
                   "read: %u, msg: %u",
                   received_len, msg_header->msg_len);
        sdc_err = SDC_DAEMON_REQUEST_REJECTED;
        goto out;

    } else if (msg_header->operation == MSG_INSTALL_KEY &&
               msg_header->msg_len <
               sizeof(sdc_daemon_msg_header_type) +
               sizeof(sdc_daemon_msg_install_type)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Message too short for install key op.");
        sdc_err = SDC_DAEMON_REQUEST_REJECTED;
        goto out;

    } else if (msg_header->operation == MSG_REMOVE_KEY &&
               msg_header->msg_len !=
               sizeof(sdc_daemon_msg_header_type) +
               sizeof(sdc_daemon_msg_remove_type)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Message length wrong for remove key op.");
        sdc_err = SDC_DAEMON_REQUEST_REJECTED;
        goto out;
    }

    /* Get the operation specific data */
    switch (msg_header->operation) {
    case MSG_INSTALL_KEY:
        msg_install = (sdc_daemon_msg_install_type *)( msg_buf + sizeof(sdc_daemon_msg_header_type));

        /* Validate the install message data */
        if (msg_install->flags & (~MSG_FLAG_ALL)) {
            daemon_log(DAEMON_LOG_ERROR,
                       "Bad key installation flags.");
            sdc_err = SDC_DAEMON_REQUEST_REJECTED;
            goto out;

        } else {
            expected_len = sizeof(sdc_daemon_msg_header_type) +
                           sizeof(sdc_daemon_msg_install_type);
            key_data = NULL;
            if (0 == (msg_install->flags & MSG_FLAG_RANDOM_KEY)) {
                expected_len += msg_install->key_data_len;

                /* Get the key data */
                key_data = (uint8_t *) (msg_buf +
                                        sizeof(sdc_daemon_msg_header_type) +
                                        sizeof(sdc_daemon_msg_install_type));
            }
            if (msg_header->msg_len != expected_len) {
                daemon_log(DAEMON_LOG_ERROR,
                           "Key length does not match "
                           "expected message length");
                sdc_err = SDC_DAEMON_REQUEST_REJECTED;
                goto out;
            }
        }
        break;

    case MSG_IMPORT_KEY:
        msg_import = (sdc_daemon_msg_import_type *)( msg_buf + sizeof(sdc_daemon_msg_header_type));

        /* Validate the install message data */
        if (msg_import->flags & (~MSG_FLAG_VOLATILE)) {
            daemon_log(DAEMON_LOG_ERROR,
                       "Bad key installation flags.");
            sdc_err = SDC_DAEMON_REQUEST_REJECTED;
            goto out;

        } else {
            expected_len = sizeof(sdc_daemon_msg_header_type) +
                           sizeof(sdc_daemon_msg_import_type);
            key_data = NULL;
            if (0 == (msg_import->flags & MSG_FLAG_RANDOM_KEY)) {
                expected_len += msg_import->key_data_len;

                /* Get the key data */
                key_data = (uint8_t *) (msg_buf +
                                        sizeof(sdc_daemon_msg_header_type) +
                                        sizeof(sdc_daemon_msg_import_type));
            }
            if (msg_header->msg_len != expected_len) {
                daemon_log(DAEMON_LOG_ERROR,
                           "Key length does not match "
                           "expected message length");
                sdc_err = SDC_DAEMON_REQUEST_REJECTED;
                goto out;
            }
        }
        break;

    case MSG_REMOVE_KEY:
        msg_remove = (sdc_daemon_msg_remove_type *) (msg_buf + sizeof(sdc_daemon_msg_header_type));

        break;

    default:
        daemon_log(DAEMON_LOG_ERROR, "Unrecognized operation type: %u.",
                   msg_header->operation);
        sdc_err = SDC_DAEMON_REQUEST_REJECTED;
        goto out;
    }

    /* Log the request */
    log_request(msg_header, cr.uid, msg_install, msg_import, msg_remove);

    /* If valid message, call LibKeyman */
    sdc_err = perform_op(msg_header, msg_install, msg_import, msg_remove,
                         key_data, ctx, &id, &cr);

out:
    if (msg_buf) {
        keystore_overwrite_secret(msg_buf, msg_buf_len);
        free(msg_buf);
    }

    /* Prepare response message to send to client */
    msg_response.msg_len = sizeof(sdc_daemon_msg_response_type);

    if (sdc_err == SDC_OK) {
        /* Operation succeded */
        msg_response.type = MSG_SUCCESS;
        msg_response.value.installed_id = id;
    } else {
        /* Error encountered */
        msg_response.type = MSG_ERROR;
        msg_response.value.sdc_error = sdc_err;
    }

    /* Send response to client */
    send_len = send(cs_fd, &msg_response, msg_response.msg_len, MSG_NOSIGNAL);
    if (send_len < 0) {
        daemon_log(DAEMON_LOG_ERROR, "write() errno: %u.", errno);
    } else if ((size_t)send_len != msg_response.msg_len) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Sent incomplete response to client");
    }
}

void sdc_handle_socket_event(int epoll_fd, int ss_fd, struct epoll_event *events, keystore_ctx_t *ctx)
{
    int err_chk;

    /* An error has occured on this fd, or the socket is not
       ready for reading (why were we notified then?) */
    if ((events->events & EPOLLERR) ||
        (!(events->events & EPOLLIN))) {
        daemon_log(DAEMON_LOG_ERROR, "epoll error: %s\n", strerror(errno));
        close(events->data.fd);
        return;
    }else if ((events->events & EPOLLHUP) ||
              (!(events->events & EPOLLIN))) {
        daemon_log(DAEMON_LOG_ERROR, "Client closed socket without communication");
        close(events->data.fd);
        return;
    }
    /* Add incoming connections from clients to epoll */
    else if (events->data.fd == ss_fd) {
        while (1) {
            struct epoll_event event;
            struct sockaddr in_name;
            socklen_t in_name_len;
            int in_fd;

            in_name_len = sizeof(in_name);
            /* PRQA: Lint Message 64: Lint is not able to handle union definition in socket.h which becomes effective when defining _GNU_SOURCE */
            in_fd = accept(ss_fd, (struct sockaddr *)&in_name, &in_name_len); /*lint !e64 */
            if (in_fd == -1) {
                if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
                    /* Finished processing all incoming connections */
                    break;
                }else  {
                    daemon_log(DAEMON_LOG_ERROR,
                               "Accept returned unexpected error %s",
                               strerror(errno));
                    break;
                }
            }

            event.data.fd = in_fd;
            event.events = EPOLLIN;
            err_chk = epoll_ctl(epoll_fd, EPOLL_CTL_ADD, in_fd, &event);
            if (err_chk == -1) {
                daemon_log(DAEMON_LOG_ERROR, "Error: epoll_ctl: %s",
                           strerror(errno));
                return;
            }
        }
    } else {
        /* Handle incoming client communication */
        sdc_handle_socket_request(events->data.fd, ctx);
        close(events->data.fd);
    }
}

/**
 * The main daemon loop that listens on a socket and handles requests
 *
 * \param ctx		LibKeyman context
 * \param socket_fd	File descriptor of socket
 * \return SDC error code if error encountered in setting up server socket
 */
sdc_error_t sdc_socket_loop(keystore_ctx_t *ctx, int socket_fd)
{
    int epoll_fd;
    int err_chk;
    struct epoll_event event;
    struct epoll_event events[MAX_EVENTS];
    int n;
    int i;

    epoll_fd = epoll_create1(0);
    if (epoll_fd == -1) {
        daemon_log(DAEMON_LOG_ERROR, "EPOLL_CREATE FAILED");
        return SDC_INTERNAL_ERROR;
    }

    event.events = EPOLLIN;
    event.data.fd = socket_fd;

    err_chk = epoll_ctl(epoll_fd, EPOLL_CTL_ADD, socket_fd, &event);
    if (err_chk == -1) {
        daemon_log(DAEMON_LOG_ERROR, "Error setting up epoll: %s", strerror(errno));
        return SDC_INTERNAL_ERROR;
    }

    /* Start the daemon loop */
    while (terminate == 0) {
        n = epoll_wait(epoll_fd, events, MAX_EVENTS, -1);

        /* Handle all incoming events */
        for (i = 0; i < n; i++) {
            sdc_handle_socket_event(epoll_fd, socket_fd, &events[i], ctx);
        }
    }

    if (terminate_with_err != 0) {
        return SDC_UNKNOWN_ERROR;
    }

    return SDC_OK;
}

/**
 * Setup the SDC socket for communication to libsdc clients
 *
 * \param listen_addr   Path to address to bind server socket to
 * \param socked_fd     Pointer to return the socket file descriptor
 * \return SDC error code if error encountered in setting up server socket
 */
sdc_error_t sdc_socket_setup (const char *listen_addr, int *socked_fd)
{
    int on;
    int err_chk;
    struct sockaddr_un addr;
    char *path_copy;
    mode_t u_mode;
    sdc_error_t sdc_err;

    memset(&addr, 0, sizeof(struct sockaddr_un));

    /* Make a copy of path, as dirname may modify the string */
    path_copy = strdup(listen_addr);
    if (path_copy == NULL) {
        return SDC_NO_MEM;
    }

    /* Create directory if it doesn't exist */
    sdc_err = sdc_helper_create_absolute_dir(dirname(path_copy), true, true);
    free(path_copy);
    if (sdc_err != SDC_OK) {
        daemon_log(DAEMON_LOG_ERROR, "socket path couldn't be created");
        return sdc_err;
    }

    /* Create the socket */
    u_mode = umask( UMASK_SOCKET );
    *socked_fd = socket(PF_LOCAL, SOCK_STREAM | O_NONBLOCK, 0);
    if (*socked_fd < 0) {
        daemon_log(DAEMON_LOG_ERROR, "Couldn't open server socket.");
        return SDC_DAEMON_SOCKET_CREATION_FAILED;
    }

    /* Let the daemon be restarted, if a previous instance of it has
     * already bound to the address, by unlinking the address.
     * TODO: Is there a better way to do this?
     */

    if (!access(listen_addr, F_OK)) {
        err_chk = unlink(listen_addr);
        if (err_chk == -1) {
            daemon_log(DAEMON_LOG_INFO, "unlink() errno: %u.", errno);
        }
    }

    /* Set socket options, to allow socket address to be reused */
    on = 1;
    err_chk = setsockopt(*socked_fd, SOL_SOCKET, SO_REUSEADDR,
                         &on, sizeof(on));
    if (err_chk == -1) {
        daemon_log(DAEMON_LOG_ERROR, "setsockopt() errno: %u.", errno);
        return SDC_DAEMON_SOCKET_CREATION_FAILED;
    }

    /* Set up the address (path) to listen to */
    addr.sun_family = AF_LOCAL;
    strncpy(addr.sun_path, listen_addr, UNIX_PATH_MAX);

    /* Bind server socket to address */
    /* PRQA: Lint Message 64: Lint is not able to handle union definition in socket.h which becomes effective when defining _GNU_SOURCE */
    err_chk = bind(*socked_fd, (struct sockaddr *) &addr, strlen(addr.sun_path) + sizeof(addr.sun_family)); /*lint !e64 */
    if (err_chk == -1) {
        daemon_log(DAEMON_LOG_ERROR, "bind() errno: %u.", errno);
        umask(u_mode);
        return SDC_DAEMON_SOCKET_CREATION_FAILED;
    }
    umask(u_mode);

    /* Listen for connections to server socket */
    err_chk = listen(*socked_fd, 50);
    if (err_chk == -1) {
        daemon_log(DAEMON_LOG_ERROR,
                   "listen() errno: %u.", errno);
        return SDC_DAEMON_COMMUNICATION_ERROR;
    }

    return SDC_OK;
}

/**
 * Cleanup the SDC socket for communication to libsdc clients
 *
 * \param listen_addr   Path to address to bind server socket to
 * \param socked_fd socket file descriptor
 * \return SDC error code if error encounter while cleaning up the socket
 */
sdc_error_t sdc_socket_cleanup (const char *listen_addr, int socked_fd)
{
    /* Remove socket */
    if (socked_fd >= 0)
        close(socked_fd);
    if (listen_addr)
        unlink(listen_addr);

    return SDC_OK;
}

/**
 * Main entry point from OS
 *
 * \param argc		Unused number of CLI args
 * \param argv		Unused CLI args
 * \return response to OS
 */
int main(int argc, char **argv)
{
    sdc_error_t err, err_sdc_ver, err_key_ver, tmp_err;
    keystore_ctx_t ctx;
    bool ctx_initialized = false;
    int socket_fd = -1;
    char *key_store = NULL;
    char *listen_addr = NULL;
    char *version_info;

    (void)argc;
    (void)argv;

    /* check versions of kernel interfaces */
    err_sdc_ver = sdc_kernel_version_verify(&version_info);
    if ((err_sdc_ver == SDC_OK) || (err_sdc_ver == SDC_INVALID_VERSION)) {
        daemon_log(DAEMON_LOG_INFO, "%s\n", version_info);
    }
    free (version_info);

    if (err_sdc_ver != SDC_OK) {
        if (err_sdc_ver == SDC_INVALID_VERSION) {
            daemon_log(DAEMON_LOG_ERROR,
                       "libSDC kernel interface version is not compatible to libSDC.");
        } else {
            daemon_log(DAEMON_LOG_ERROR,
                       "libSDC version check failed.");
        }
    }

    err_key_ver = keyman_kernel_version_verify(&version_info);
    if ((err_key_ver == SDC_OK) || (err_key_ver == SDC_INVALID_VERSION)) {
        daemon_log(DAEMON_LOG_INFO, "%s\n", version_info);
    }
    free (version_info);

    if (err_key_ver != SDC_OK) {
        if (err_key_ver == SDC_INVALID_VERSION) {
            daemon_log(DAEMON_LOG_ERROR,
                       "libKeyman kernel interface version is not compatible to libKeyman.");
        } else {
            daemon_log(DAEMON_LOG_ERROR,
                       "libKeyman version check failed.");
        }
    }

    if ((err_sdc_ver != SDC_OK) || (err_key_ver != SDC_OK)) {
        fprintf(stderr, "KHD exited with error %d(%s)\n",
                err_sdc_ver, sdc_get_error_string(err_sdc_ver));

        return err_sdc_ver;
    }

    /* Initialize paths from config file, or defaults */
    err = init_paths(&key_store, &listen_addr);
    if (err != SDC_OK) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Could not initialise paths.");
    }

    if (err == SDC_OK) {
        /* Initialize Keystore context */
        err = keystore_ctx_init(&ctx, key_store);
        if (err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR,
                       "Couldn't initialise Keyman library: %s",
                       sdc_get_error_string(err));
        } else {
            ctx_initialized = true;
        }
    }


    if (err == SDC_OK) {
        err = keystore_initialize(&ctx);
        if ((err != SDC_OK) &&
            (err != SDC_KEY_STORAGE_ALREADY_PRESENT)) {
            daemon_log(DAEMON_LOG_ERROR,
                       "Daemon failed to create key-store - Aborting");
        } else {
            if (err == SDC_OK) {
                daemon_log(DAEMON_LOG_IMPORTANT_INFO,
                           "New keystore created.");
            }
            err = SDC_OK;
        }
    }

    if (err == SDC_OK) {
        err = keystore_verify_credentials(&ctx);
        if (err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR, "Daemon failed to verify key-store credentials - %s - Aborting",
                       sdc_get_error_string(err));
        }
    }


    /* Load keys stored on disc */
    if (err == SDC_OK) {
        err = keystore_restore_persistent(&ctx);
        if (err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR,
                       "Daemon failed to load keys from key-store - %s - Aborting",
                       sdc_get_error_string(err));
        }
    }


    if (err == SDC_OK) {
        /* check if keys shall be imported from key container */
        err = key_container_check_and_import(&ctx);
        if (err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR, "Daemon failed to import key containers - %s - Aborting",
                       sdc_get_error_string(err));
        }
    }

    if (err == SDC_OK) {
        /* check if keys shall be imported from key container */
        err = keystore_unlock(&ctx);
        if (err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR, "Daemon failed to unlock the keystore - %s - Aborting",
                       sdc_get_error_string(err));
        }
    }

    if (err == SDC_OK) {
        install_signal_handler();

        /* Do the main loop */
        err = sdc_socket_setup(listen_addr, &socket_fd);
    }

    if (err == SDC_OK) {
        /* Notify systemd that we have finished startup and are ready to handle
           connections */
        sd_notify(0, "READY=1");

        err = sdc_socket_loop(&ctx, socket_fd);
    }

    tmp_err = sdc_socket_cleanup(listen_addr, socket_fd);
    if (err == SDC_OK) {
        err = tmp_err;
    }

    /* Close keystore context */
    if (ctx_initialized)
        keystore_ctx_fini(&ctx);
    /* free pointers if not already NULL */
    free(listen_addr);
    free(key_store);

    if (err != SDC_OK)
        fprintf(stderr, "KHD exited with error %d(%s)\n",
                err, sdc_get_error_string(err));

    return err;
}

